
# EcoMUX.py
import os
import sys
import json
import subprocess
import time
import signal

class EcoMUX:
    def __init__(self, π):
        self.π = π
        self.meta_path = os.path.expanduser('~/.eco/emux.meta'.replace('/', os.sep))
        self.port_base = 64000
        self.meta = self.load_meta()

    def load_meta(self):
        if not os.path.exists(self.meta_path):
            return {}
        with open(self.meta_path, 'r', encoding='utf-8') as f:
            return json.load(f)

    def save_meta(self):
        os.makedirs(os.path.dirname(self.meta_path), exist_ok=True)
        with open(self.meta_path, 'w', encoding='utf-8') as f:
            json.dump(self.meta, f, indent=4)

    def get_next_port(self):
        used_ports = [info.get('port', 0) for info in self.meta.values()]
        port = self.port_base
        while port in used_ports:
            port += 1
        return port

    def new(self, label):
        if label in self.meta:
            print(f"[emux] ❌ Session '{label}' already exists!")
            return

        cwd = os.getcwd()
        start_time = time.time()
        port = self.get_next_port()
        host_id = str(int(time.time()))
        salt = str(random.randint(30, 100))

        cmd = [
            sys.executable,
            os.path.abspath(__file__),
            '-label', label,
            '-parent', host_id,
            '-salt', salt
        ]

        process = subprocess.Popen(
            cmd,
            cwd=cwd,
            stdout=subprocess.DEVNULL,
            stderr=subprocess.DEVNULL,
            stdin=subprocess.DEVNULL,
            start_new_session=True
        )

        self.meta[label] = {
            'pid': process.pid,
            'cwd': cwd,
            'start_time': start_time,
            'port': port,
            'host_id': host_id,
            'salt': salt
        }
        self.save_meta()
        print(f"[emux] 🚀 Started '{label}' on port {port}")

    def kill(self, label):
        if label not in self.meta:
            print(f"[emux] ❌ No session '{label}'.")
            return
        try:
            os.kill(self.meta[label]['pid'], signal.SIGKILL)
        except Exception as e:
            print(f"[emux] ⚠️ Kill error: {e}")
        self.meta.pop(label, None)
        self.save_meta()

    def list(self):
        if not self.meta:
            print("[emux] ❌ No active sessions.")
            return
        for label, info in self.meta.items():
            print(f"- {label}: PID {info['pid']} Port {info['port']}")

π.e.EcoMUX=EcoMUX
