import datetime
def friendlyDate( theDate ):
	return str(datetime.datetime.fromtimestamp(float(theDate)).strftime('%Y-%m-%d %H:%M:%S'))
import datetime
import time

def autoDate(data, fail=False):
    try:
        import datefinder # type: ignore
        matches = list(datefinder.find_dates(str(data)))
        if matches:
            return matches[0].timestamp()
    except Exception:
        pass

    if isinstance(data, (int, float)):
        if int(data) > 1000000000:
            if int(data) > 9999999999:
                return int(data) / 1000
            return int(data)
        else:
            return False

    try:
        s = str(data).strip()
        s = s.replace('T', ' ').split('+')[0].split('Z')[0].strip()
        if any(x in s.lower() for x in ['am', 'pm']):
            try:
                dt = datetime.datetime.strptime(s, '%Y-%m-%d %I:%M:%S %p')
                return dt.timestamp()
            except Exception:
                pass
            try:
                dt = datetime.datetime.strptime(s, '%Y-%m-%d %I:%M %p')
                return dt.timestamp()
            except Exception:
                pass

        # Try ISO standard
        try:
            dt = datetime.datetime.fromisoformat(s)
            return dt.timestamp()
        except Exception:
            pass

        # Try some common formats
        for fmt in [
            '%Y-%m-%d %H:%M:%S',
            '%Y-%m-%d %H:%M',
            '%Y-%m-%d',
            '%m/%d/%Y',
            '%m/%d/%y',
            '%d-%m-%Y',
            '%d/%m/%Y',
            '%b %d %Y',
            '%B %d %Y',
            '%b %d, %Y',
            '%B %d, %Y',
        ]:
            try:
                dt = datetime.datetime.strptime(s, fmt)
                return dt.timestamp()
            except Exception:
                continue

        # If still looks numeric
        if s.isdigit():
            if len(s) > 10:
                return int(s) / 1000
            return int(s)

    except Exception as e:
        if fail:
            raise e
        return False

    if fail:
        raise Exception('Could not parse date')
    return False
import time
from datetime import datetime, timedelta

def ago(text):
    dt = autoDate(text)
    if dt:
        return dt
    text = text.strip().lower()
    now = time.time()
    
    # Fast path: if it ends with 'm', assume month first
    if text.endswith('m') and text[:-1].isdigit():
        number = int(text[:-1])
        now_dt = datetime.fromtimestamp(now)
        year = now_dt.year
        month = now_dt.month + number

        while month > 12:
            month -= 12
            year += 1

        days_in_month = [31,
                         29 if year % 4 == 0 and (year % 100 != 0 or year % 400 == 0) else 28,
                         31,30,31,30,31,31,30,31,30,31]
        day = min(now_dt.day, days_in_month[month - 1])
        new_dt = datetime(year, month, day, now_dt.hour, now_dt.minute, now_dt.second)
        return int(new_dt.timestamp())

    units = {
        's': 1,
        'sec': 1,
        'secs': 1,
        'second': 1,
        'seconds': 1,
        'min': 60,
        'mins': 60,
        'minute': 60,
        'minutes': 60,
        'h': 3600,
        'hr': 3600,
        'hrs': 3600,
        'hour': 3600,
        'hours': 3600,
        'd': 86400,
        'day': 86400,
        'days': 86400,
        'w': 604800,
        'wk': 604800,
        'wks': 604800,
        'week': 604800,
        'weeks': 604800,
        'y': 31536000,
        'yr': 31536000,
        'yrs': 31536000,
        'year': 31536000,
        'years': 31536000,
    }

    import re
    match = re.match(r'^(\d+)\s*([a-z]+)$', text)
    if not match:
        raise ValueError(f"Invalid format: {text}")
    
    number = int(match.group(1))
    unit = match.group(2)

    unit = unit.rstrip('s')  # allow plural forms like "hours", "minutes"

    if unit not in units:
        raise ValueError(f"Unknown time unit: {unit}")
    
    seconds = number * units[unit]
    return int(now + seconds)


import sys

class SwitchManager:
    def __init__(self, command=None, Switches=None, Triggers=None):

        if type(command) == str:
            while '  ' in command: command = command.replace('  ', ' ')
            command = command.split(' ')

        if Switches is None:
            Switches = {}
        if Triggers is None:
            Triggers = {}

        self.defaultTriggers = {
            'Ago': π.e.ago,
        }
        self.triggers = {}
        for key in self.defaultTriggers:
            self.triggers[key] = self.defaultTriggers[key]
        for key in Triggers:
            self.triggers[key] = Triggers[key]

        self.defaultSwitches = {}

        self.switchesRegister = {}
        for key in Switches:
            self.switchesRegister[key] = Switches[key]
        for key in self.defaultSwitches:
            if key not in self.switchesRegister:
                self.switchesRegister[key] = self.defaultSwitches[key]

        if not command:
            command = sys.argv
        self.command = command
        self.app = command[0]
        self.args = command[1:]

        self.used = {}
        self._Values = {}
        self.usage = {}
        self.instances = {}

        # Clean up switchesRegister formatting
        for key in self.switchesRegister:
            self.used[key] = False
            text = self.switchesRegister[key]
            text = text.replace(',', ' ').replace('|', ' ').replace(';', ' ')
            while '  ' in text:
                text = text.replace('  ', ' ')
            self.switchesRegister[key] = text.strip().replace(' ', ',')

        # Initialize empty value holders
        for key in self.switchesRegister:
            self._Values[key] = []

        # Build a flag lookup
        self.flag_to_key = {}
        for key, flags in self.switchesRegister.items():
            for flag in flags.split(','):
                self.flag_to_key[flag] = key

        self.parse()

    def strip(self):
        clean = []
        for item in self.command:
            if item not in self.switchesRegister and item not in self.flag_to_key:
                clean.append(item)
        return clean


    def parse(self):
        current_switch = None
        current_key = None
        i = 0

        while i < len(self.args):
            arg = self.args[i]
            if arg in self.flag_to_key:
                current_key = self.flag_to_key[arg]
                current_switch = arg

                if current_key not in self.instances:
                    self.used[current_key] = True
                    self.instances[current_key] = {}

                if current_switch not in self.instances[current_key]:
                    self.instances[current_key][current_switch] = []

                if current_key not in self.usage:
                    self.usage[current_key] = []
                if current_switch not in self.usage[current_key]:
                    self.usage[current_key].append(current_switch)

                if self._Values[current_key] == []:
                    self._Values[current_key] = True  # Assume True first
            else:
                if current_key and current_switch:
                    if self._Values[current_key] is True:
                        self._Values[current_key] = []
                    if current_key in self.triggers:
                        value = self.triggers[current_key](arg)
                    else:
                        value = arg
                    self._Values[current_key].append(value)
                    self.instances[current_key][current_switch].append(value)
            i += 1

    def isActive(self, name):
        return self.used.get(name, False)

    def Values(self, name, instance=None):
        if name not in self.instances:
            return []
        if instance is not None:
            return self.instances[name].get(instance, [])
        return self.values(name)

    def values(self, name):
        if name not in self._Values:
            return []
        if self._Values[name] is True:
            return []
        return self._Values[name]
    
    def value(self, name):
        val = self.values(name)
        return ','.join(val)
    
    def validate(self):
        import json

        print('\nApp:')
        print(self.app)

        print('\nUsed:')
        print(json.dumps(self.used, indent=4))

        print('\nValues:')
        print(json.dumps(self._Values, indent=4))
        if 'Ago' in self._Values:
            for ago in self._Values['Ago']:
                print('Ago:', π.e.friendlyDate(ago))

        print('\nUsage:')
        print(json.dumps(self.usage, indent=4))

        print('\nInstances:')
        print(json.dumps(self.instances, indent=4))
π.e.friendlyDate=friendlyDate
π.e.autoDate=autoDate
π.e.ago=ago
π.e.SwitchManager=SwitchManager
