
import os
import time
import duckdb # type: ignore
from cryptography.hazmat.primitives.asymmetric import rsa # type: ignore
from cryptography.hazmat.primitives import serialization # type: ignore

class EcoKeyManager:
	def __init__(self, eco_private_db):
		"""
		eco_private_db: Instance of EcoPrivateDB class (already opened).
		"""
		self.private = eco_private_db
		self.private_table = 'eco_keys'
		self.setup_table()

	def setup_table(self):
		self.private.conn.execute(f"""
			CREATE TABLE IF NOT EXISTS {self.private_table} (
				id INTEGER PRIMARY KEY,
				label TEXT,
				private_key TEXT,
				public_key TEXT,
				created_at TEXT
			)
		""")

	def generate_keys(self, label='default'):
		"""
		Generate new RSA keypair and store inside private.db
		"""
		private_key = rsa.generate_private_key(
			public_exponent=65537,
			key_size=4096,
		)
		public_key = private_key.public_key()

		private_pem = private_key.private_bytes(
			encoding=serialization.Encoding.PEM,
			format=serialization.PrivateFormat.PKCS8,
			encryption_algorithm=serialization.NoEncryption()
		).decode('utf-8')

		public_pem = public_key.public_bytes(
			encoding=serialization.Encoding.PEM,
			format=serialization.PublicFormat.SubjectPublicKeyInfo
		).decode('utf-8')

		self.private.conn.execute(f"""
			INSERT INTO {self.private_table} (label, private_key, public_key, created_at)
			VALUES (?, ?, ?, ?)
		""", (label, private_pem, public_pem, time.strftime('%Y-%m-%d %H:%M:%S')))
		self.private.conn.commit()

	def get_private_key(self, label='default'):
		"""
		Load a private key for label
		"""
		result = self.private.conn.execute(f"""
			SELECT private_key FROM {self.private_table} WHERE label=?
		""", (label,)).fetchone()
		if not result:
			raise Exception(f"No private key found for label '{label}'")
		return serialization.load_pem_private_key(
			result[0].encode('utf-8'),
			password=None,
		)

	def get_public_key(self, label='default'):
		"""
		Load a public key for label
		"""
		result = self.private.conn.execute(f"""
			SELECT public_key FROM {self.private_table} WHERE label=?
		""", (label,)).fetchone()
		if not result:
			raise Exception(f"No public key found for label '{label}'")
		return serialization.load_pem_public_key(
			result[0].encode('utf-8')
		)

	def delete_keypair(self, label='default'):
		"""
		Remove a keypair safely
		"""
		self.private.conn.execute(f"""
			DELETE FROM {self.private_table} WHERE label=?
		""", (label,))
		self.private.conn.commit()

	def list_keypairs(self):
		"""
		List available key labels
		"""
		results = self.private.conn.execute(f"""
			SELECT label, created_at FROM {self.private_table}
		""").fetchall()
		return [{'label': r[0], 'created_at': r[1]} for r in results]
π.e.EcoKeyManager=EcoKeyManager
