<?php

class AppManager {
    private $basePath;
    private $md5Salt;

    public function __construct($basePath, $md5Salt) {
        $this->basePath = rtrim($basePath, '/') . '/';
        $this->md5Salt = $md5Salt;
    }

    public function generateMd5($input) {
        return md5($input . $this->md5Salt);
    }

    // public function getAppId_old($url) {
    //     $cleanUrl = preg_replace('/\?.*/', '', $url);
    //     return $this->generateMd5($cleanUrl);
    // }

    public function getAppId($url) {
        $cleanUrl = preg_replace('/\?.*/', '', $url);
        return $this->generateMd5($cleanUrl);
    }

    public function getFolderId($path) {
        return $this->generateMd5($path);
    }

    public function getFileId($path) {
        return $this->generateMd5($path);
    }

    public function readJson($path) {
        $fullPath = $this->basePath . $path;
        if (file_exists($fullPath)) {
            return json_decode(file_get_contents($fullPath), true);
        }
        return null;
    }

    public function writeJson($path, $data) {
        $fullPath = $this->basePath . $path;
        $dir = dirname($fullPath);
        if (!is_dir($dir)) {
            mkdir($dir, 0777, true);
        }
        file_put_contents($fullPath, json_encode($data, JSON_PRETTY_PRINT));
    }

    public function inflateAccess($access) {
        if (is_array($access)) {
            $access = implode(',', $access);
        }
    
        $accessing = [
            'admin' => 'admin',
            'v' => 'view',
            'view' => 'view',
            'a' => 'add',
            'add' => 'add',
            'm' => 'modify',
            'mod' => 'modify',
            'modify' => 'modify',
            'mv' => 'move',
            'move' => 'move',
            'e' => 'edit',
            'ed' => 'edit',
            'edit' => 'edit',
            'd' => 'dir',
            'dir' => 'dir',
            'c' => 'checklist',
            'check' => 'checklist',
            'ch' => 'checklist',
            'checklist' => 'checklist',
            'de' => 'delete',
            'del' => 'delete',
            'delete' => 'delete',
            'php' => 'php',
        ];
    
        $accessList = [];
        foreach (explode(',', $access) as $a) {
            if (isset($accessing[$a])) {
                $accessList[] = $accessing[$a];
            }
        }
        return $accessList;
    }
    
    

    private function parseUrl($url) {
        $parsedUrl = parse_url($url);
        $query = [];
        if (isset($parsedUrl['query'])) {
            parse_str($parsedUrl['query'], $query);
        }
        return $query;
    }

    public function autoID($identifier) {
        if (strlen($identifier) === 32 && ctype_xdigit($identifier)) {
            return $identifier; // MD5 hash
        } else {
            // Strip all non-numeric characters and format phone number
            $numericPhone = preg_replace('/\D/', '', $identifier);
            return $this->generateMd5($numericPhone);
        }
    }

    public function addFolder($identifier, $url, $folderPath = '', $access = '') {
        // Validate and extract the GET argument if provided
        $folderPath = $this->validateAndExtractGetArg($folderPath, 'fo');
    
        if (!$folderPath) {
            throw new Exception("Folder path must be specified.");
        }
    
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $folderId = $this->getFolderId($folderPath);
    
        $this->addAppInfo($url);
    
        $folderInfo = [
            "path" => $folderPath,
            "access" => $this->inflateAccess($access)
        ];
    
        $this->writeJson("members/$memberId/apps/$appId/folders/$folderId.json", $folderInfo);
    }
    

    public function addFile($identifier, $url, $filePath = '', $access = '', $pin = '') {
        // Validate and extract the GET argument if provided
        $filePath = $this->validateAndExtractGetArg($filePath, 'f');
    
        if (!$filePath) {
            throw new Exception("File path must be specified.");
        }
    
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $fileId = $this->getFileId($filePath);
    
        $this->addAppInfo($url);
    
        $fileInfo = [
            "path" => $filePath,
            "access" => $this->inflateAccess($access),
            "pin" => $pin // Add pin to file info
        ];
    
        $this->writeJson("members/$memberId/apps/$appId/files/$fileId.json", $fileInfo);
    }
    
    public function activePin($memberId, $path) {
        $fileId = $this->getFileId($path);
        // echo $fileId; exit;
        $appId = $this->appSearch($memberId,$fileId);
    
        $fileInfo = $this->readJson("members/$memberId/apps/$appId/files/$fileId.json");
        // echo "members/$memberId/apps/$appId/files/$fileId.json";
        // var_dump($fileInfo);
        return isset($fileInfo['pin']) ? $fileInfo['pin'] : null;
    }
    
    

    public function removeFolder_old($identifier, $url, $folderPath = '') {
        $query = $this->parseUrl($url);
        if (!$folderPath && isset($query['fo'])) {
            $folderPath = $query['fo'];
        }
        if (!$folderPath) {
            throw new Exception("Folder path must be specified.");
        }

        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $folderId = $this->getFolderId($folderPath);

        $fullPath = $this->basePath . "members/$memberId/apps/$appId/folders/$folderId.json";
        if (file_exists($fullPath)) {
            unlink($fullPath);
        }
    }


    public function removeFolder($identifier, $appId, $folderPath = '') {
        // $query = $this->parseUrl($url);
        if (!$folderPath && isset($query['fo'])) {
            $folderPath = $query['fo'];
        }
        if (!$folderPath) {
            throw new Exception("Folder path must be specified.");
        }

        $memberId = $this->autoID($identifier);
        // $appId = $this->getAppId($url);
        $folderId = $this->getFolderId($folderPath);

        $fullPath = $this->basePath . "members/$memberId/apps/$appId/folders/$folderId.json";
        if (file_exists($fullPath)) {
            unlink($fullPath);
        }
    }

    public function removeFile_old($identifier, $url, $filePath = '') {
        $query = $this->parseUrl($url);
        if (!$filePath && isset($query['f'])) {
            $filePath = $query['f'];
        }
        if (!$filePath) {
            throw new Exception("File path must be specified.");
        }

        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $fileId = $this->getFileId($filePath);

        $fullPath = $this->basePath . "members/$memberId/apps/$appId/files/$fileId.json";
        if (file_exists($fullPath)) {
            unlink($fullPath);
        }
    }

    public function removeFile($identifier, $appId, $filePath = '') {
        // $query = $this->parseUrl($url);
        if (!$filePath && isset($query['f'])) {
            $filePath = $query['f'];
        }
        if (!$filePath) {
            throw new Exception("File path must be specified.");
        }

        $memberId = $this->autoID($identifier);
        // $appId = $this->getAppId($url);
        $fileId = $this->getFileId($filePath);

        $fullPath = $this->basePath . "members/$memberId/apps/$appId/files/$fileId.json";
        if (file_exists($fullPath)) {
            unlink($fullPath);
        }
    }

    public function addMember($userInfo) {
        $memberId = $this->autoID($userInfo['phone']);
        $this->writeJson("members/$memberId/who.json", $userInfo);
    }

    public function removeMember($identifier) {
        $memberId = $this->autoID($identifier);
        $fullPath = $this->basePath . "members/$memberId";
        if (is_dir($fullPath)) {
            $this->deleteDirectory($fullPath);
        }
    }

    public function add($jsonFile) {
        $data = $this->readJson($jsonFile);

        if (isset($data['userInfo'])) {
            $this->addMember($data['userInfo']);
        }

        if (isset($data['folders'])) {
            foreach ($data['folders'] as $folder) {
                $this->addFolder($data['userInfo']['phone'], $folder['url'], $folder['path'], $folder['access'] ?? '');
            }
        }

        if (isset($data['files'])) {
            foreach ($data['files'] as $file) {
                $this->addFile($data['userInfo']['phone'], $file['url'], $file['path'], $file['access'] ?? '');
            }
        }
    }

    private function deleteDirectory_old($dir) {
        if (!file_exists($dir)) {
            return true;
        }

        if (!is_dir($dir)) {
            return unlink($dir);
        }

        foreach (scandir($dir) as $item) {
            if ($item == '.' || $item == '..') {
                continue;
            }

            if (!$this->deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {
                return false;
            }
        }

        return rmdir($dir);
    }

    private function deleteDirectory_old2($dir) {
        if (!file_exists($dir)) {
            return;
        }
        if (!is_dir($dir)) {
            unlink($dir);
            return;
        }
        foreach (scandir($dir) as $item) {
            if ($item === '.' || $item === '..') continue;
            $this->deleteDirectory($dir . DIRECTORY_SEPARATOR . $item);
        }
        rmdir($dir);
    }

    public function checkAccess($identifier, $url, $filePath = '', $folderPath = '') {
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);

        if ($filePath) {
            $fileId = $this->getFileId($filePath);
            $fileInfo = $this->readJson("members/$memberId/apps/$appId/files/$fileId.json");
            if ($fileInfo && isset($fileInfo['access'])) {
                return $fileInfo['access'];
            }
        }

        if ($folderPath) {
            $folderId = $this->getFolderId($folderPath);
            $folderInfo = $this->readJson("members/$memberId/apps/$appId/folders/$folderId.json");
            if ($folderInfo && isset($folderInfo['access'])) {
                return $folderInfo['access'];
            }
        }

        $appInfo = $this->readJson("members/$memberId/apps/$appId/info.json");
        if ($appInfo && isset($appInfo['access'])) {
            return $appInfo['access'];
        }

        $memberInfo = $this->readJson("members/$memberId/who.json");
        if ($memberInfo && isset($memberInfo['access'])) {
            return $memberInfo['access'];
        }

        return null;
    }

    // New functions
    public function folderList($identifier, $url) {
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);

        $foldersPath = $this->basePath . "members/$memberId/apps/$appId/folders";
        $folders = [];
        if (is_dir($foldersPath)) {
            $folderFiles = scandir($foldersPath);
            foreach ($folderFiles as $file) {
                if ($file === '.' || $file === '..') {
                    continue;
                }
                $folderInfo = $this->readJson("members/$memberId/apps/$appId/folders/$file");
                if ($folderInfo) {
                    $folders[] = $folderInfo;
                }
            }
        }
        return $folders;
    }

    public function fileList($identifier, $url) {
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);

        $filesPath = $this->basePath . "members/$memberId/apps/$appId/files";
        $files = [];
        if (is_dir($filesPath)) {
            $fileFiles = scandir($filesPath);
            foreach ($fileFiles as $file) {
                if ($file === '.' || $file === '..') {
                    continue;
                }
                $fileInfo = $this->readJson("members/$memberId/apps/$appId/files/$file");
                if ($fileInfo) {
                    $files[] = $fileInfo;
                }
            }
        }
        return $files;
    }

    public function appSearch($memberId, $id) {
        $appsPath = $this->basePath . "members/$memberId/apps";
        $apps = scandir($appsPath);
        // print_r($appsPath);
        foreach ($apps as $appId) {
            if ($appId === '.' || $appId === '..') {
                continue;
            }

            $filePath = "$appsPath/$appId/files/$id.json";
            $folderPath = "$appsPath/$appId/folders/$id.json";


            // echo $filePath . "<br>";

            if (file_exists($filePath) || file_exists($folderPath)) {
                return $appId;
            }
        }

        // return '-err-';
        return null;
    }



    public function appList() {
        $appsPath = $this->basePath . "apps";
        $apps = scandir($appsPath);
        $appList = [];

        foreach ($apps as $appId) {
            if ($appId === '.' || $appId === '..') {
                continue;
            }

            $appInfoPath = "$appsPath/$appId/info.json";
            if (file_exists($appInfoPath)) {
                $appInfo = $this->readJson("apps/$appId/info.json");
                if ($appInfo && isset($appInfo['url'])) {
                    $appList[$appId] = $appInfo['url'];
                }
            }
        }

        return $appList;
    }

    public function addAppInfo($url) {
        $appId = $this->getAppId($url);
        $cleanUrl = preg_replace('/\?.*/', '', $url);
        $appInfo = [
            "url" => $cleanUrl
        ];
        $this->writeJson("apps/$appId/info.json", $appInfo);
    }
    



    
    

    
    public function getBasePath() {
        return $this->basePath;
    }
    
    public function setAppAccess($memberId, $appId, $access) {
        $accessData = $this->inflateAccess($access);
        $appMemberPath = "members/$memberId/apps/$appId/access.json";
        $this->writeJson($appMemberPath, $accessData);
    }
    

    public function getAppAccess($memberId, $appId) {
        $appMemberPath = "members/$memberId/apps/$appId/access.json";
        $accessData = $this->readJson($appMemberPath);
        return $accessData ? $accessData : [];
    }
    
    public function appGetAdd($memberId, $url, $key, $value) {
        $appId = $this->getAppId($url);
        $keyValueId = md5($key . $value);
        $appGetPath = "members/$memberId/apps/$appId/get/$keyValueId.json";
        $getInfo = [
            "key" => $key,
            "value" => $value
        ];
        $this->writeJson($appGetPath, $getInfo);
    }
    
    

    public function appGetRemove($memberId, $url, $key, $value) {
        $appId = $this->getAppId($url);
        $keyValueId = md5($key . $value);
        $appGetPath = "members/$memberId/apps/$appId/get/$keyValueId.json";
        if (file_exists($appGetPath)) {
            unlink($appGetPath);
        }
    }
    
    
    public function appGetList($memberId, $url) {
        $appId = $this->getAppId($url);
        $getListPath = "members/$memberId/apps/$appId/get";
        $getList = [];
    
        if (is_dir($getListPath)) {
            $getFiles = scandir($getListPath);
            foreach ($getFiles as $getFile) {
                if ($getFile !== '.' && $getFile !== '..') {
                    $getInfo = $this->readJson("$getListPath/$getFile");
                    if ($getInfo) {
                        $getList[] = $getInfo;
                    }
                }
            }
        }
    
        return $getList;
    }
    
    
    
    public function addMemberAppUrl_old($memberId, $url) {
        // Validate the URL format
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            throw new Exception("Invalid URL: $url");
        }
    
        // Optionally, ensure the URL starts with "https://"
        if (!preg_match('/^https:\/\//', $url)) {
            throw new Exception("URL must start with 'https://': $url");
        }
    
        // Strip query parameters for the app ID
        $cleanUrl = preg_replace('/\?.*/', '', $url);
        $appId = $this->getAppId($cleanUrl);
    
        // Add app info
        $appInfo = ["url" => $cleanUrl];
        $this->writeJson("apps/$appId/info.json", $appInfo);
        
        // Ensure the member's app directory exists
        $memberAppPath = "members/$memberId/apps/$appId";
        if (!is_dir($memberAppPath)) {
            mkdir($memberAppPath, 0777, true);
        }
    
        // Write info.json for the member's app
        $memberAppInfo = ["url" => $cleanUrl];
        $this->writeJson("$memberAppPath/info.json", $memberAppInfo);
    }
    
    public function addMemberAppUrl($memberId, $url, $access) {
        $cleanUrl = preg_replace('/\?.*/', '', $url);
        $appId = $this->getAppId($cleanUrl);
        $this->setAppAccess($memberId, $appId, $access);
        $appInfo = ["url" => $cleanUrl, "access" => $this->inflateAccess($access)];
        $this->writeJson("apps/$appId/info.json", $appInfo);
        
        // Ensure the member's app directory exists
        $memberAppPath = "members/$memberId/apps/$appId";
        if (!is_dir($memberAppPath)) {
            mkdir($memberAppPath, 0777, true);
        }
    
        // Write info.json for the member's app
        $memberAppInfo = ["url" => $cleanUrl, "access" => $this->inflateAccess($access)];
        $this->writeJson("$memberAppPath/info.json", $memberAppInfo);
    }
    
    

    public function setAppGetAccess($memberId, $url, $key, $value, $access) {
        $appId = $this->getAppId($url);
        $keyValueId = md5($key . $value);
        $accessData = $this->inflateAccess($access);
        $appGetAccessPath = "members/$memberId/apps/$appId/get_access/$keyValueId.json";
        $this->writeJson($appGetAccessPath, $accessData);
    }
    
    
    
    
    
    

    public function getAppGetAccess($memberId, $url, $key, $value) {
        $appId = $this->getAppId($url);
        $keyValueId = md5($key . $value);
        $appGetAccessPath = "members/$memberId/apps/$appId/get_access/$keyValueId.json";
        $accessData = $this->readJson($appGetAccessPath);
        return $accessData ? $accessData : [];
    }
    
    
    
    
    public function listFileAccess($memberId, $url, $filePath) {
        $appId = $this->getAppId($url);
        $fileId = $this->getFileId($filePath);
        $fileAccessPath = "members/$memberId/apps/$appId/files_access/$fileId.json";
        return $this->readJson($fileAccessPath);
    }

    public function listFolderAccess($memberId, $url, $folderPath) {
        $appId = $this->getAppId($url);
        $folderId = $this->getFolderId($folderPath);
        $folderAccessPath = "members/$memberId/apps/$appId/folders_access/$folderId.json";
        return $this->readJson($folderAccessPath);
    }

    public function listGetAccess($memberId, $url, $key) {
        $appId = $this->getAppId($url);
        $getAccessPath = "members/$memberId/apps/$appId/get_access/$key.json";
        return $this->readJson($getAccessPath);
    }

    public function listGetAccessWithValue($memberId, $url, $key, $value) {
        $appId = $this->getAppId($url);
        $getAccessPath = "members/$memberId/apps/$appId/get_access/$key.json";
        $accessData = $this->readJson($getAccessPath);
        
        return $accessData ? $accessData : [];
    }
    
    public function appListForMember($memberId) {
        $appsPath = $this->basePath . "members/$memberId/apps";
        $apps = scandir($appsPath);
        $appList = [];
    
        foreach ($apps as $appId) {
            if ($appId === '.' || $appId === '..') {
                continue;
            }
    
            $appInfoPath = "$appsPath/$appId/info.json";
            if (file_exists($appInfoPath)) {
                $appInfo = $this->readJson($appInfoPath);
                if ($appInfo && isset($appInfo['url'])) {
                    $appList[] = $appInfo['url'];
                }
            }
        }
    
        return $appList;
    }
    
    public function removeApp($memberId, $appId) {
        // Check if appId is provided
        if (empty($appId)) {
            throw new Exception("App ID is required");
        }
    
        // Path to the member's app directory
        $memberAppPath = "members/$memberId/apps/$appId";
    
        // Ensure the directory exists
        if (is_dir($memberAppPath)) {
            // Remove the directory and its contents
            $this->deleteDirectory($memberAppPath);
        }
    
        // Optionally, remove the app info if it's no longer associated with any member
        $allMembers = scandir('members');
        $isAppUsed = false;
        foreach ($allMembers as $member) {
            if ($member === '.' || $member === '..') continue;
            if (is_dir("members/$member/apps/$appId")) {
                $isAppUsed = true;
                break;
            }
        }
    
        if (!$isAppUsed) {
            // Remove the app info
            $appInfoPath = "apps/$appId/info.json";
            if (file_exists($appInfoPath)) {
                unlink($appInfoPath);
            }
        }
    }
    
    // Utility function to delete a directory and its contents
    private function deleteDirectory($dir) { //new
        if (!is_dir($dir)) {
            return;
        }
        $objects = scandir($dir);
        foreach ($objects as $object) {
            if ($object != "." && $object != "..") {
                if (is_dir($dir . "/" . $object) && !is_link($dir . "/" . $object)) {
                    $this->deleteDirectory($dir . "/" . $object);
                } else {
                    unlink($dir . "/" . $object);
                }
            }
        }
        rmdir($dir);
    }
    
    public function validateAndExtractGetArg($path, $getArg) {
        // Check if the path starts with "https://"
        if (strpos($path, 'https://') === 0) {
            // Parse the URL to get the query parameters
            $parsedUrl = parse_url($path);
            if (isset($parsedUrl['query'])) {
                parse_str($parsedUrl['query'], $queryParams);
                // Check if the GET argument exists in the query parameters
                if (isset($queryParams[$getArg])) {
                    return $queryParams[$getArg];
                }
            }
        }
        // Return the path as it is if it doesn't start with "https://" or the GET argument is not found
        return $path;
    }
    

}
