<?php
// Fetch database credentials
function getDBCredentials() {
    $credentials = json_decode(file_get_contents('https://keys.sds.sh/?json=1&prefix=1&id=programmer-dnd-*'), true);
    return $credentials;
}

// Establish database connection
function getDBConnection() {
    $credentials = getDBCredentials();
    $conn = new mysqli(
        $credentials['host'],
        $credentials['user'],
        $credentials['password'],
        $credentials['db']
    );

    if ($conn->connect_error) {
        die('Database connection failed: ' . $conn->connect_error);
    }
    return $conn;
}

// Process incoming requests via action
if (isset($_REQUEST['action'])) {
    $action = $_REQUEST['action'];

    header('Content-Type: application/json');

    try {
        switch ($action) {
            case 'createPlayer':
                createPlayer($_REQUEST);
                break;
            case 'login':
                login($_REQUEST);
                break;
            case 'fetchAll':
                echo json_encode(fetchAll($_REQUEST['table']));
                break;
            case 'fetchById':
                echo json_encode(fetchById($_REQUEST['table'], $_REQUEST['id']));
                break;
            case 'insert':
                insert($_REQUEST['table'], $_REQUEST['data']);
                echo json_encode(['success' => true]);
                break;
            case 'update':
                update($_REQUEST['table'], $_REQUEST['data'], $_REQUEST['id']);
                echo json_encode(['success' => true]);
                break;
            case 'delete':
                delete($_REQUEST['table'], $_REQUEST['id']);
                echo json_encode(['success' => true]);
                break;
            default:
                throw new Exception("Invalid action: $action");
        }
    } catch (Exception $e) {
        echo json_encode(['error' => $e->getMessage()]);
    }

    exit;
}

// Create a new player and optional Dungeon Master
function createPlayer($data) {
    $conn = getDBConnection();

    // Insert player
    $playerData = [
        'Name' => $data['name'],
        'Email' => $data['email'],
        'Password' => null
    ];
    insert('Players', $playerData);

    // Insert Dungeon Master if selected
    if (!empty($data['isDungeonMaster']) && $data['isDungeonMaster'] === 'true') {
        $dmData = [
            'Name' => $data['name'],
            'Email' => $data['email'],
            'ExperienceLevel' => $data['experienceLevel']
        ];
        insert('DungeonMasters', $dmData);
    }

    echo json_encode(['success' => true]);
}

// Login function
function login($data) {
    $conn = getDBConnection();

    // Fetch player by email
    $sql = "SELECT * FROM Players WHERE Email = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $data['email']);
    $stmt->execute();
    $player = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$player || $player['Password'] !== $data['password']) {
        echo json_encode(['error' => 'Invalid login credentials.']);
        return;
    }

    // Check if the player is also a Dungeon Master
    $sql = "SELECT * FROM DungeonMasters WHERE Email = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $data['email']);
    $stmt->execute();
    $dm = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    $conn->close();

    if ($dm) {
        echo json_encode(['success' => true, 'isDungeonMaster' => true, 'dungeonMasterID' => $dm['DungeonMasterID']]);
    } else {
        echo json_encode(['success' => true, 'isDungeonMaster' => false]);
    }
}

// Fetch all records from a table
function fetchAll($table) {
    $conn = getDBConnection();
    $sql = "SELECT * FROM $table";
    $result = $conn->query($sql);
    $rows = [];
    while ($row = $result->fetch_assoc()) {
        $rows[] = $row;
    }
    $conn->close();
    return $rows;
}

// Fetch a single record by primary key
function fetchById($table, $id) {
    $conn = getDBConnection();

    $primaryKey = determinePrimaryKey($table);

    $sql = "SELECT * FROM $table WHERE $primaryKey = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    $conn->close();

    return $result;
}

// Insert a new record into a table
function insert($table, $data) {
    $conn = getDBConnection();
    $fields = implode(", ", array_keys($data));
    $placeholders = implode(", ", array_fill(0, count($data), '?'));
    $types = str_repeat('s', count($data));

    $sql = "INSERT INTO $table ($fields) VALUES ($placeholders)";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...array_values($data));

    if (!$stmt->execute()) {
        die("Execution failed: " . $stmt->error);
    }

    $stmt->close();
    $conn->close();
}

// Update an existing record by primary key
function update($table, $data, $id) {
    $conn = getDBConnection();

    $primaryKey = determinePrimaryKey($table);

    $fields = implode(" = ?, ", array_keys($data)) . " = ?";
    $types = str_repeat('s', count($data)) . 'i';

    $sql = "UPDATE $table SET $fields WHERE $primaryKey = ?";
    $stmt = $conn->prepare($sql);
    $params = array_merge(array_values($data), [$id]);
    $stmt->bind_param($types, ...$params);

    if (!$stmt->execute()) {
        die("Execution failed: " . $stmt->error);
    }

    $stmt->close();
    $conn->close();
}

// Delete a record by primary key
function delete($table, $id) {
    $conn = getDBConnection();

    $primaryKey = determinePrimaryKey($table);

    $sql = "DELETE FROM $table WHERE $primaryKey = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);

    if (!$stmt->execute()) {
        die("Execution failed: " . $stmt->error);
    }

    $stmt->close();
    $conn->close();
}

// Determine the primary key for a table
function determinePrimaryKey($table) {
    return match ($table) {
        'DungeonMasters' => 'DungeonMasterID',
        'Battles' => 'BattleID',
        'Campaigns' => 'CampaignID',
        'Characters' => 'CharacterID',
        'Players' => 'PlayerID',
        'Items' => 'ItemID',
        'CharacterSpells' => 'CharacterSpellID',
        'Narratives' => 'NarrativeID',
        'Storylines' => 'StorylineID',
        'GameManager' => 'GameManagerID',
        'GamePlayers' => 'GamePlayerID',
        default => 'id',
    };
}
?>